/*
 * FILE:         $Source: /cvs_archive/cvs/blast/blastConfig.h,v $
 * 
 * DESCRIPTION:  blastConfig: class that holds program settings read from rc file and 
 *               from command-line
 *               Settings are:
 *               BLASTHOST: url of the blast website, default www.ncbi.nlm.nih.gov
 *               BLASTPORT: server port, 80
 *               BLASTPROG: CGI script name, blast.cgi
 *               COMMENTFILE: str comment file, /bmrb/lib/comment.str
 *
 * AUTHOR:       $Author: dmaziuk $
 *
 * DATE:         $Date: 2001/03/15 22:33:35 $
 *
 * UPDATE HISTORY
 * --------------
 * $Log: blastConfig.h,v $
 * Revision 2.2  2001/03/15 22:33:35  dmaziuk
 * Disabled deposition of PDB entries in molecular system description frame.
 *
 * Revision 2.1  2001/01/17 22:52:35  dmaziuk
 * Fixed a bug where wrong sequence length was deposited into starfile.
 *
 * Revision 2.0  2001/01/11 23:44:43  dmaziuk
 * Version 2 of blast contains complete rewrite of blastParser and major rewrite
 * of (almost) everything else.
 *
 */
#ifndef BLASTCONFIG_H
#define BLASTCONFIG_H

#include <iostream.h>
#include <fstream.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef IS_IRIX
#  include <mstring.h>
#endif

#ifdef IS_GCC
#  include <string>
#endif

#include "ast.h"
#include "constants.h"

// blastConfig class
/** This class stores program settings, constants & static functions. 
    <p>
    Settings that are loaded from blastrc file: <br>
    url (blasthost), port number, CGI script name and comment file name. <br> 
    Settings that are read from command line: <br>
    "quiet" flag (-q), "sequence details" flag (-seqdet). <br>
    Other: temp. file name to store query result, map of database names and codes.
    <p>
    BlastConfig provides Java-style accessor methods: getXXX/setXXX/isXXX. 
  */
class blastConfig
{
  private:
    // these map db names to db codes (see .cc for definition)
    static const string fDbNames[];
    static const string fDbCodes[];

  public:
    /// number of databases (10)
    static const int NumDbs = 10;
    /// index for PDB database (0)
    static const int pdb    = 0;
    /// index for DBJ database
    static const int dbj    = 1;
    /// index for EMBL database
    static const int emb    = 2;
    /// index for GenBank (GB) database
    static const int gb     = 3;
    /// index for GenBank (GI) database
    static const int gi     = 4;
    /// index for PIR database
    static const int pir    = 5;
    /// index for PRF database
    static const int prf    = 6;
    /// index for REF database
    static const int ref    = 7;
    /// index for SWISS-PROT database
    static const int sp     = 8;
    /// index for unknown database (9)
    static const int none   = 9;

    /// number of columns in data entry
    static const int NumCols = 8;

    /// start of data section in query result (see .cc for definition)
    static const string strFlag;
    /// end of data section in query result (see .cc for definition)
    static const string endFlag;

    /// max. number of non-pdb entries to keep
    static const int MaxXEntries = 5;

    /// "Length =" with 10 spaces in front is the end of entries
    static const int lengthOffset = 10;
           
    // db name/code map
    /** returns db name or NULL.<br>
        It is overloaded to accept either string (code) 
        or int (index) parameter (pdb..none).
	@see pdb
      */
    static string getName( const string& code )
        { for( int i = 0; i < blastConfig::NumDbs; i++ ) 
            if( fDbCodes[i] == code ) return fDbNames[i]; 
          return ""; 
        };
    static string getName( const int& index ) { return fDbNames[index]; };

    /** returns index of db name or -1.
        @see pdb
      */
    static int getNameIdx( const string& name )
        { for( int i = 0; i < blastConfig::NumDbs; i++ )
            if( fDbNames[i] == name ) return i;
          return -1;
        };
    /** returns db code or NULL.<br>
        It is overloaded to accept either string (name) 
        or int (index) parameter (pdb..none).
	@see pdb
      */
    static string getCode( const string& name )
        { for( int i = 0; i < blastConfig::NumDbs; i++ ) 
            if( fDbNames[i] == name ) return fDbCodes[i]; 
          return ""; 
        };
    static string getCode( const int& index ) { return fDbCodes[index]; };

    /** returns index of db name or -1.
        @see pdb
     */
    static int getCodeIdx( const string& code )
        { for( int i = 0; i < blastConfig::NumDbs; i++ ) 
            if( fDbCodes[i] == code ) return i; 
          return -1; 
        };

    /// returns URL of the blast server
    inline char *getBlastHost() const { return (char *) fBlastHost; }
    /// returns CGI script name
    inline char *getBlastProg() const { return (char *) fBlastProg; }
    /// returns port number on the blast server
    inline unsigned getBlastPort() const { return fBlastPort; }
    /// returns name of comment file
    inline char *getCommentFile() const { return (char *) fCommentFile; }
    /// returns path to temp. file where query result is saved
    inline char *getQueryFile() const { return (char *) fQueryFile; }
    /// returns path to log file
    inline char *getLogFile() const { return (char *) fLogFile; }
    /// returns true if blast was started with "quiet" ("-q") flag
    inline bool isQuiet() const { return fQuiet; }
    /// sets "quiet" flag
    inline void setQuiet( bool q ) { fQuiet = q; }
    /// returns true if blast was started with "seq.details" ("-seqdet") flag
    inline bool isSeqDet() const { return fSeqDet; }
    /// sets "sequence details" flag
    inline void setSeqDet( bool q ) { fSeqDet = q; }
    
    /** returns quote style constant for the string. <br>
        Quote styles are defined in starlib (ast.h), they're just
        an enum of none, single quote, double quote or semicolon 
        (for multi-line entries).
	<p>
        It is static so we don't need an object to call it.
        @param searchme string to check.
        @return quote style.
      */ 
    static DataValueNode::ValType findNeededQuoteStyle( const string& searchme );

    /** constructor. <br>
        Set blast host, program etc. to compile-time defaults, <br>
        try opening blastrc file and read new values from it, <br>
        see if TMPDIR envvar is set, <br>
          and set temporary file to $TMPDIR/blast_out, <br>
	  otherwise set it to P_tmpdir/blast_out 
        (P_tmpdir is defined in stdio.h)
      */
    blastConfig();

   private:
    char fBlastHost[MAX_STRLEN];
    char fBlastProg[MAX_STRLEN];
    unsigned fBlastPort;
    char fCommentFile[MAX_STRLEN];
    char fQueryFile[MAX_STRLEN];
    char fLogFile[MAX_STRLEN];
    bool fQuiet;
    bool fSeqDet;

};

#endif // BLASTCONFIG_H
