/*
 * FILE:         $Source: /cvs_archive/cvs/blast/dataEntry.h,v $
 * 
 * DESCRIPTION:  declarations for class dataEntry
 *
 * AUTHOR:       $Author: dmaziuk $
 *
 * DATE:         $Date: 2001/01/17 22:52:36 $
 *
 * UPDATE HISTORY
 * --------------
 * $Log: dataEntry.h,v $
 * Revision 2.1  2001/01/17 22:52:36  dmaziuk
 * Fixed a bug where wrong sequence length was deposited into starfile.
 *
 * Revision 2.0  2001/01/11 23:44:44  dmaziuk
 * Version 2 of blast contains complete rewrite of blastParser and major rewrite
 * of (almost) everything else.
 *
 */
#ifndef DATAENTRY_H
#define DATAENTRY_H

#include <iostream.h>
#include <iomanip.h>
#include <stdlib.h>
#include <string.h>

#ifdef IS_IRIX
#  include <mstring.h>
#  include <vector.h>
#endif

#ifdef IS_GCC
#  include <string>
#  include <vector>
#endif

#include "constants.h"
#include "blastConfig.h"

class dataEntries;

// dataEntry class
/** class for holding data entries.
 *  <p>
 *  Each <code>dataEntry</code> object holds one data record. <br>
 *  All fields are private, they are accessed via Java-style (getX/setX/isX etc) 
 *  methods.<br>
 *  There are operators <code><</code>, <code>></code> and <code>==</code> for
 *  sorting the vector of <code>dataEntry</code> objects and for finding duplicate
 *  entries. <br>
 *  <b>Note</b> that mening of operators <code><</code> and <code>></code>
 *  is somewhat counter-intuitive, RTFM carefully before using them.
 */
class dataEntry
{
    friend class dataEntries;
    
  private:
    int fLength;                 // "Length = 104" -> 104
// "Identities = 101/103 (98%)" is split into 
    int fSeqLen;                 // sequence length       = 101 
    int fComp;                   // sequence completeness = 103 / res. count * 100
    int fId;                     // sequence id           = 98 (%)
// "Positives = 101/103 (98%)" -- BLASTP only
    int fPos;                    // 98 -- only percentage is used 
    int fIdCount;                // count of "Identities" lines
    string fExp;                 // "Expect = 4e-54" -> 4e-54
    string fName;                // database name
    string fCode;                // accession code
    string fMolName;             // molecular system description
    string fDetails;             // for "-seqdet" flag
    bool fPullOut;
    dataEntries *fParent;        // in case we need access to it
    blastConfig *fConfig;        // to access "quiet" and "seqdet" flags
    
  public:
    /** constructor. 
     *  <p>
     *  Initialize member variables, set up pointers to <code>blastConfig</code>
     *  and <code>dataEntries</code> objects.
     *  @param d pointer to <code>dataEntries</code> vector.
     *  @param b pointer to <code>blastConfig</code> object.
     *  @see dataEntries
     *  @see blastConfig
     */
    dataEntry( dataEntries *d, blastConfig *b );
    
    /** destructor.
     *  <p>
     *  Erase string fields, just in case.
     */
    ~dataEntry();

    /** used by STL <code>sort</code> algorithm.
     *  <p>
     *  Since we want entries sorted by source (db name) with pdb on top, 
     *  and then in descending order of identity, this operator returns true if <br>
     *  <code>this.fName < other.fName</code> or <br>
     *  <code>this.fName == other.fName</code> and <code>this.fId > other.fId</code>.<br>
     *  See <code>blastConfig</code> class for <code>fName</code> ordering 
     *  (<code>blastConfig::pdb</code> etc.)
     *  @return true or false
     *  @see blastConfig
     */
    virtual bool operator< ( const dataEntry &entry );

    /** This operator is not used.
     *  <p>
     *  operator <code>></code> returns true if <br>
     *  <code>this.fName > other.fName</code> or <br>
     *  <code>this.fName == other.fName</code> and <code>this.fId < other.fId</code> <br>
     *  (IOW it's a complement to <code><</code>, see <code>operator <</code> for explanation).
     *  @return true or false
     */
    virtual bool operator> ( const dataEntry &entry );

    /** used to search for duplicate entries by <code>dataEntries::add()</code>.
     *  <p>
     *  operator <code>==</code> returns true if <br>
     *  <code>this.fName == other.fName</code> and <code>this.fCode == other.fCode</code>.
     *  @return true or false
     *  @see dataEntries
     */
    virtual bool operator== ( const dataEntry &entry );

    /** return pointer to <code>dataEntries</code> vector.
     *  @see dataEntries
     */
    inline dataEntries *getParent() const { return fParent; }
    /** Set up the pointer to <code>dataEntries</code> vector.
     *  <b>Note</b> use constructor instead.
     */
    inline void setParent( dataEntries *parent ) { fParent = parent; }

    /** return pointer to <code>blastConfig</code> object.
     *  @see blastConfig
     */
    inline blastConfig *getConfig() const { return fConfig; }
    /** Set up the pointer to <code>blastConfig</code> object.
     *  <b>Note</b> use constructor instead.
     */
    inline void setConfig( blastConfig *config ) { fConfig = config; }

    /// return length (eg. 104)
    inline int getLength() const { return fLength; }
    /// set length
    inline void setLength( const int& len ) { fLength = len; }

    /// return sequence length (eg. 101)
    inline int getSeqLen() const { return fSeqLen; }
    /// set sequence length
    inline void setSeqLen( const int& len ) { fSeqLen = len; }

    /// return sequence completeness (calculated from identities and residue count)
    inline int getComp() const { return fComp; }
    /// set sequence completeness
    inline void setComp( const int& comp ) { fComp = comp; }

    /// return id (eg. 98%)
    inline int getId() const { return fId; }
    /// set id
    inline void setId( const int& id ) { fId = id; }

    /// return positives (eg. 98%)
    inline int getPos() const { return fPos; }
    /// set positives
    inline void setPos( const int& pos ) { fPos = pos; }

    /// return count of "Identities" lines
    inline int getIdCount() const { return fIdCount; }
    /// set count of "Identities" lines
    inline void setIdCount( const int& cnt ) { fIdCount = cnt; }

    /// return "Expect" value
    inline string getExp() const { return fExp; }
    /** set "Expect" value.
     *  Overloaded to accept a <code>string</code> or <code>char *</code> parameter.
     */
    inline void setExp( const char *expect ) { fExp = expect; }
    inline void setExp( const string& expect ) { fExp = expect; } 

    /// return database name (eg. PDB)
    inline string getName() const { return fName; }
    /** set database name.
     *  Overloaded to accept a <code>string</code> or <code>char *</code> parameter.
     */
    inline void setName( const char *name ) { fName = name; }
    inline void setName( const string& name ) { fName = name; } 

    /// return accession code
    inline string getCode() const { return fCode; }
    /** set database accession code.
     *  Overloaded to accept a <code>string</code> or <code>char *</code> parameter.
     */
    inline void setCode( const char *code ) { fCode = code; }
    inline void setCode( const string& code ) { fCode = code; } 

    /// return molecular system description
    inline string getMolName() const { return fMolName; }
    /** set molecular system description.
     *  Overloaded to accept a <code>string</code> or <code>char *</code> parameter.
     */
    inline void setMolName( const char *mol ) { fMolName = mol; }
    inline void setMolName( const string& mol ) { fMolName = mol; } 

    /// return sequence details
    inline string getDetails() const { return fDetails; }
    /** set sequence details.
     *  Overloaded to accept a <code>string</code> or <code>char *</code> parameter.
     */
    inline void setDetails( const char *details ) { fDetails = details; }
    inline void setDetails( const string& details ) { fDetails = details; } 

    /// return pull-out flag: true or false.
    inline bool getPullOut() { return fPullOut; }
    /// set pull-out flag.
    inline void setPullOut( bool b ) { fPullOut = b; }

    /** print object to <code>cout</code>. <br>
     *  Used for debugging.
     */
    void print();

};

#endif // DATAENTRY_H
