/*
 * FILE:               $Source: /cvs_archive/cvs/blast/main.cc,v $
 * 
 * DESCRIPTION:        this is the main function for blast
 * 
 * AUTHOR:             $Author: dmaziuk $
 * 
 * DATE:               $Date: 2001/03/15 22:33:35 $
 * 
 * UPDATE HISTORY:
 * ---------------
 * $Log: main.cc,v $
 * Revision 2.2  2001/03/15 22:33:35  dmaziuk
 * Disabled deposition of PDB entries in molecular system description frame.
 *
 * Revision 2.1  2001/02/20 21:00:08  dmaziuk
 * Few minor changes
 *
 */
#include "main.h"

StarFileNode *AST;
ofstream *os;
//ofstream logstr;

/** Program's mainline.
   <p>
   1. Parse command-line arguments. <br>
   2. Read input starfile. <br>
   3. Read coment file. <br>
   4. Find saveframes that contain _Mol_residue_sequence. <br>
   5. Query BLAST site. <br>
   6. Compare query result with existing frames. <br>
   7. If different, deposit new data into starfile.
   @return 0 = OK, non-zero value otherwise.
   @see blastConfig
   @see blastSearch
   @see blastParser
   @see blastDeposit
   @see dataEntries
  */
int main(  int argc, char *argv[]  ) 
{
  blastConfig bl_config;
  blastSearch bl_search( &bl_config );
  blastParser bl_parser( &bl_config );
  blastDeposit bl_deposit( &bl_config );
  
  char outFileName[MaxFileName];

  List<ASTnode*> *result;
  SaveFrameNode *save_frame;

  StarFileNode *commentStarTree;

  star_warning_check = false;  //turn off warning
  star_syntax_print = false;   //turn off warning

  bool overwrite = false;
  FILE *inputFile = (FILE *)NULL;
  char inputFileName[MaxFileName];
  os = (ofstream *)NULL;
  limitId = 98; //default lower limit of id

  // iterate over the command-line argument
  // usage: blast [-q] [-ow] [-seqdet] <inputfile> [id]
  //-------------------------------------------------------------------------- 
    if( argc < 2 ) 
    {
	cerr << "Usage: blast [-q] [-ow] [-seqdet] <inputfile> [id]" << endl;
	exit( 1 );
    }
    
    for( int argNum = 1; argNum < argc; argNum++ )
    {
        // "-q" means "quiet mode"(reduce output message) for script
        if( !strcmp( argv[argNum], "-q" ) )
            bl_config.setQuiet( true );

        // "-ow" means "overwrite input file with blast output"
        else if( !strcmp( argv[argNum], "-ow" ) )
            overwrite = true;

        // -seqdet means "put in the sequence details tag"
        else if( strcmp(argv[argNum], "-seqdet" ) == 0 )
            bl_config.setSeqDet( true );

        // go on to input file
        else if( inputFile == NULL )
        {
            strcpy( inputFileName, argv[argNum] );
            if( (inputFile = fopen( argv[argNum], "r" )) == NULL )
            { 
                cerr << "***** Error: opening input starfile \"" ;
                cerr << inputFileName << "\"" << endl;
                exit( 1 );
            }
        }

       // if loop reaches this far, get the lower limit of id
       else
       {
           limitId = atoi( argv[argNum] );
           if( (limitId > 100) || (limitId < 0) )
           {
               cout << "***** Error: lower limit of Id should be between 0 to 100.\n";
               exit( 1 );
           }
       }
    }//end of for

//    logstr.open( bl_config.getLogFile() ); 
//    if( !logstr ) cerr << "Cannot open " << bl_config.getLogFile() << endl;
    
    // reads input starfile into memory to parse
    //------------------------------------------
    yyin = inputFile;

    if( yyparse() != 0 )
    {
        cerr << "***** Error: syntax error while parsing \""; 
        cerr << inputFileName << "\"" << endl;
        exit(1);
    }

// since input file was successfully parsed, AST now contains it in memory.
    inputStarFile = AST;    

// now read comment file into memory to parse
    yyin = fopen( bl_config.getCommentFile(), "r" );

// error check for comment file
    if( yyin == NULL )
    {
        cerr << "***** Error: opening comment file \"";
        cerr << bl_config.getCommentFile() << "\"" << endl;
        exit( 1 );
    }

//parse the comment file into AST
    if( yyparse() != 0 )
    {
        cerr << "***** Error: syntax error while parsing file \"";
        cerr << bl_config.getCommentFile() << "\"" << endl;
        exit( 1 );
    }
    
    commentStarTree = AST;

// find save frames that contains _Mol_residue_sequence
    result = inputStarFile->searchForTypeByTag( ASTnode::SAVEFRAMENODE,
     string( "_Mol_residue_sequence" ) );
    if( result->size() == 0 )
    {
        cout << "***** Error: the starfile you entered doesn't contain residue" << endl;
        cout << "             sequence. Can't perform blast search." << endl;
        exit(1);
    }
    else 
    {
        int pdb_index = 0;
        bool revise = false;
        bool revise_pdb_list = false;

        if( !bl_config.isQuiet() )
            cout << "Commencing blast search, please wait for results." << endl;
        else
            cout << "--- Running blast on " << inputFileName << endl ;

//
        dataEntries *revise_entries;
//this loop will handle inputfile with one seq/more than one seq 
        for (int index = 0, size = 0, error = 0; size < result->size(); index++, size++)
        {
            dataEntries *ab_entries = new dataEntries();

            if( result->size() > 1 )
                cout << "For " << (size+1) << "th sequence: " << endl;          

            save_frame = (SaveFrameNode *)((*result)[index]);

            if( bl_search.do_blast_search( save_frame ) == -1 ) 
            {
	            if( result->size() == 1 ) exit( 1 );
                else 
                {
                    error++;
                    if( error == result->size() ) exit( 1 );
                    else continue;
                }
            }
            cout << " blast search done " << endl;

            if( bl_parser.parse( ab_entries, save_frame ) < 0 )
            {
                if( result->size() == 1) 
                {
                    delete ab_entries;
                    exit( 1 );
                }
                else
                {
                    error++;
                    if( error == result->size() ) 
                    {
                        delete ab_entries;
                        exit( 1 );
                    }
                    else continue;
                }
            }
            if( !bl_config.isQuiet() )
            {
                cout << "blastParser::parse() done, ";
                cout  << ab_entries->size() << " entries extracted" << endl;
            }
	    
// sort the entries by id & database name
	    ab_entries->sort();
#ifdef DEBUG
  cout << "Done sorting" << endl;
#endif

//if the blast output is different from old one, revise input file with it
            revise = bl_parser.isUpdated( ab_entries, save_frame );

// if any one of the saveframe was revised pdb_list should be revised
            if( bl_deposit.depositMonoPoly( save_frame, ab_entries, revise ) == -1 )
            {
                if (result->size() == 1)  exit(1);
                else 
                {
                    error++;
                    if (error == result->size()) exit(1);
                    else continue;
                }
            }
//HACK: if pdb list was revised we ned to make ab_entries 
// available outside the for loop  
            if( revise ) revise_entries = ab_entries;
            else delete ab_entries;
        } // endfor

// revised pdb_list is deposited in this frame
        if( revise ) 
        {
// Thu Mar 15 16:24:44 2001 dmaziuk -- disabled	at Eldon's request
//            bl_deposit.depositMolSys( *revise_entries );
            delete revise_entries;
        }
    } // endif result->size() == 0

// if "-ow" is on, overwrite input starfile with blast output
// otherwise output filename is going to be "inputFileName.blast"
    strcpy( outFileName, inputFileName );
    if( !overwrite ) strcat( outFileName, ".blast" );

    os = new ofstream(outFileName);
    if( os == NULL )
    {
        cerr << "***** Error: opening output starfile \"" << outFileName;
        cerr << "\"" << endl;
        exit(1);
    }

     //inputStarFile = input starfile, commentStarTree = comment file
     //now we run the formatter
     //--------------------------------------------------------------
    formatNMRSTAR( inputStarFile, commentStarTree );
 
    fclose(yyin);
    inputStarFile->Unparse( 0 );
    delete inputStarFile;
    delete commentStarTree;  //do not delete AST, since it is currently pointing
                             //at the same place as commentStarTree, which is 
                             //already deleted
    delete os;

    if( !bl_config.isQuiet() )
    {    
        cout << "Deposit the result into \"" << outFileName;  
        cout << "\" is done. " << endl;
    }

/*  if( logstr )
 *   {
 *	 logstr << "result deposited into " << outFileName << endl;
 *	 logstr.flush();
 *	 logstr.close();
 *   }
 */ 
    return 0;
}

